/* roach.c
   Copyright (C) 2001 Daiki Ueno

   Xroach - A game of skill.  Try to find the roaches under your windows.
   Copyright 1991 by J.T. Anderson
*/

#include "roach.h"
#include <PalmOS.h>
#include "MathLib.h"

#define ROACH_HEADINGS	24	/* number of orientations */
#define ROACH_ANGLE	15	/* angle between orientations */
#define ROACH_WIDTH	32
#define ROACH_HEIGHT	32
typedef struct RoachMap {
  float sine;
  float cosine;
} RoachMap;

typedef struct Roach {
    RoachMap *rp;
    int index;
    float x;
    float y;
    int hidden;
    int turnLeft;
    int steps;
} Roach;

RoachMap roachPix[ROACH_HEADINGS];
Roach *roaches;
int maxRoaches = 8;
int minRoaches = 3;
int curRoaches = 0;
float roachSpeed = 20.0;
int turnSpeed = 10;
Coord screenX;
Coord screenY;
Coord screenWidth;
Coord screenHeight;
Boolean squishRoach = true;

UInt16 timeoutPeriod = 0;
typedef void (*TimeoutHandlerPtr) ();
TimeoutHandlerPtr timeoutHandler = 0;

Boolean supportsRomVersion35 = false;

/* utility functions */
static void LoadMathLib ();
static int RandInt (int maxVal);
static Boolean PrefsHandleEvent (EventPtr event);
static Boolean RoachHandleEvent (EventPtr event);

static Boolean ApplicationHandleEvent (EventPtr event);
static void ApplicationDisplayDialog (UInt16 formID);
static void EventLoop ();

static void FormSetIntField (FormPtr form, UInt16 fieldID, int value);
static Boolean FormGetIntField (FormPtr form, UInt16 fieldID, int *value);
static void FormFreeField (FormPtr form, UInt16 fieldID);
static void FormClose ();

static void LoadRoaches ();
static void RoachHandleTimeout ();

/* functions stolen from xroach.c */
static void AddRoach ();
static void TurnRoach (Roach *roach);
static void MoveRoach (int rx);
static void MoveRoaches ();
static void DrawRoaches ();
static int RoachInRect (Roach *roach, int rx, int ry, int x, int y,
			unsigned int width, unsigned int height);
static void checkSquish (int x, int y);

static void
LoadMathLib ()
{
  Err error = SysLibFind (MathLibName, &MathLibRef);
  if (error)
    error = SysLibLoad (LibType, MathLibCreator, &MathLibRef);
  ErrFatalDisplayIf (error, "Can't find MathLib");
  error = MathLibOpen (MathLibRef, MathLibVersion);
  ErrFatalDisplayIf (error, "Can't open MathLib");
}

/** Draw a bitmap attached with resources at the given coordinates. */
void
DrawBitmap (id, x, y)
     int id, x, y;
{
  MemHandle bmapHandle = DmGet1Resource ('Tbmp', id);
  BitmapPtr bmap = MemHandleLock (bmapHandle);
  if (supportsRomVersion35)
    WinPaintBitmap (bmap, screenX + x, screenY + y);
  else
    WinDrawBitmap (bmap, screenX + x, screenY + y);
  MemHandleUnlock (bmapHandle);
  DmReleaseResource (bmapHandle);
}

static void
FormSetIntField (form, fieldID, value)
     FormPtr form;
     UInt16 fieldID;
     int value;
{
  FieldPtr field = FrmGetObjectPtr
    (form, FrmGetObjectIndex (form, fieldID));
  MemHandle handle = FldGetTextHandle (field);
  char *text;
  if (!handle)
    {
      handle = MemHandleNew (10);
      if (!handle)
	{
	  SysFatalAlert ("Failed to allocate memory for edit field.");
	  return;
	}
    }
  text = MemHandleLock (handle);
  StrIToA (text, value);
  MemHandleUnlock (handle);
  FldSetTextHandle (field, handle);
  FldDrawField (field);
}

static Boolean
FormGetIntField (form, fieldID, value)
     FormPtr form;
     UInt16 fieldID;
     int *value;
{
  FieldPtr field = FrmGetObjectPtr
    (form, FrmGetObjectIndex (form, fieldID));
  MemHandle handle = FldGetTextHandle (field);
  char *text;
  if (!handle)
    return false;
  text = MemHandleLock(handle);
  *value = StrAToI(text);
  MemHandleUnlock(handle);
  return true;
}

static void
FormFreeField (form, fieldID)
     FormPtr form;
     UInt16 fieldID;
{
  FieldPtr field = FrmGetObjectPtr
    (form, FrmGetObjectIndex (form, fieldID));
  MemHandle handle = FldGetTextHandle (field);
  if (handle)
    {
      FldSetTextHandle (field, NULL);
      MemHandleFree (handle);
    }
}

static void
FormClose ()
{
  EventType event;
  MemSet (&event, sizeof (EventType), 0);
  event.eType = frmCloseEvent;
  event.data.frmClose.formID = FrmGetActiveFormID ();
  EvtAddEventToQueue (&event);
}

/** Generate random integer between 0 and maxVal-1. */
static int
RandInt (maxVal)
     int maxVal;
{
  return SysRandom (0) % maxVal;
}

static void
LoadRoaches ()
{
  int ax;
  for (ax = 0; ax < 360; ax += ROACH_ANGLE)
    {
      int rx = ax / ROACH_ANGLE;
      float angle = rx * 0.261799387799;
      RoachMap *rp = &roachPix[rx];
      rp->sine = sin (angle);
      rp->cosine = cos (angle);
    }
}

/** Check for roach completely in specified rectangle. */
static int
RoachInRect (roach, rx, ry, x, y, width, height)
     Roach *roach;
     int rx;
     int ry;
     int x;
     int y;
     unsigned int width;
     unsigned int height;
{
  if (rx < x)
    return 0;
  if (rx + ROACH_WIDTH > x + width)
    return 0;
  if (ry < y)
    return 0;
  if (ry + ROACH_HEIGHT > y + height)
    return 0;
  return 1;
}

/** Give birth to a roach. */
static void
AddRoach ()
{
  Roach *roach = &roaches[curRoaches++];
  roach->index = RandInt (ROACH_HEADINGS);
  roach->rp = &roachPix[roach->index];
  roach->x = RandInt (screenWidth - ROACH_WIDTH);
  roach->y = RandInt (screenHeight - ROACH_HEIGHT);
  roach->hidden = 0;
  roach->steps = RandInt (turnSpeed);
  roach->turnLeft = RandInt (100) >= 50;
}

/** Turn a roach. */
static void
TurnRoach (roach)
     Roach *roach;
{
  if (roach->index != roach->rp - roachPix)
    return;

  if (roach->turnLeft)
    {
      roach->index += (RandInt (30) / 10) + 1;
      if (roach->index >= ROACH_HEADINGS)
	roach->index -= ROACH_HEADINGS;
    }
  else
    {
      roach->index -= (RandInt (30) / 10) + 1;
      if (roach->index < 0)
	roach->index += ROACH_HEADINGS;
    }
}

/** Move a roach. */
static void
MoveRoach (rx)
     int rx;
{
  Roach *roach;
  float newX;
  float newY;

  roach = &roaches[rx];
  newX = roach->x + (roachSpeed * roach->rp->cosine);
  newY = roach->y - (roachSpeed * roach->rp->sine);
    
  if (RoachInRect (roach, (int)newX, (int)newY, 0, 0,
		   screenWidth, screenHeight))
    {
      roach->x = newX;
      roach->y = newY;

      if (roach->steps-- <= 0)
	{
	  TurnRoach (roach);
	  roach->steps = RandInt (turnSpeed);
	  if (RandInt (100) >= 80)
	    roach->turnLeft ^= 1;
	}
    }
  else
    TurnRoach (roach);
}

/** Move all roaches. */
static void
MoveRoaches ()
{
  int rx;
  for (rx = 0; rx < curRoaches; rx++)
    {
      Roach *roach = &roaches[rx];
      if (!roach->hidden)
	{
	  RectangleType rect;
	  RctSetRectangle (&rect,
			   screenX + (int)roach->x, screenY + (int)roach->y,
			   ROACH_WIDTH, ROACH_HEIGHT);
	  WinEraseRectangle (&rect, 0);
	  MoveRoach (rx);
	}
    }
}

/** Draw all roaches. */
static void
DrawRoaches ()
{
  WinDrawOperation current = winPaint; 
  int rx;
  if (supportsRomVersion35)
    current = WinSetDrawMode (winOverlay);
  for (rx = 0; rx < curRoaches; rx++)
    {
      Roach *roach = &roaches[rx];
      if (roach->hidden)
	continue;
      roach->rp = &roachPix[roach->index];
      DrawBitmap (RoachBitmapFirst + roach->index,
		  (int)roach->x, (int)roach->y);
    }
  if (supportsRomVersion35)
    WinSetDrawMode (current);
}

/** Check to see if we have to squish any roaches. */
static void
checkSquish (x, y)
     int x, y;
{
    int i, intX, intY;
    int rx;
    for (rx = 0; rx < curRoaches; rx++)
      {
	Roach *roach = &roaches[rx];
	if (roach->hidden)
	  continue;
	intX = (int)roach->x;
	intY = (int)roach->y;
	if (x > intX &&
	    x < intX + ROACH_WIDTH &&
	    y > intY &&
	    y < intY + ROACH_HEIGHT)
	  {
	    SndCommandType s;
	    s.cmd = sndCmdFreqDurationAmp;
	    s.param1 = 150;
	    s.param2 = 20;
	    s.param3 = sndMaxAmp;
	    SndDoCmd (0, &s, 0);
	    DrawBitmap (SquishBitmap, intX, intY);
	    for (i = rx; i < curRoaches - 1; i++)
	      roaches[i] = roaches[i + 1];
	    curRoaches--;
	    rx--;
	  }
      }
}

static inline void
StartRoach ()
{
  timeoutPeriod = SysTicksPerSecond () / 10;
  timeoutHandler = RoachHandleTimeout;
}

static inline void
StopRoach ()
{
  timeoutPeriod = 0;
  timeoutHandler = 0;
}

static Boolean
RoachHandleEvent (event)
     EventPtr event;
{
  FormPtr form;
  Boolean processed = false;
  switch (event->eType)
    {
    case frmOpenEvent:
      form = FrmGetActiveForm ();
      FrmDrawForm (form);
      StartRoach ();
      processed = true;
      break;
    case menuEvent:
      switch (event->data.menu.itemID)
	{
	case MenuItemAbout:
	  form = FrmInitForm (AboutForm);
	  FrmDoDialog (form);
	  processed = true;
	  break;
	case MenuItemPrefs:
	  ApplicationDisplayDialog (PrefsForm);
	  processed = true;
	  break;
	default:
	  break;
	}
      break;
    case penDownEvent:
      if (!squishRoach)
	break;
      checkSquish (event->screenX - screenX, event->screenY - screenY);
      processed = true;
      break;
    default:
      break;
    }
  return processed;
}

static Boolean
PrefsHandleEvent (event)
     EventPtr event;
{
  Boolean processed = false, error = false;
  FormPtr form = FrmGetActiveForm ();
  int value;
  switch (event->eType)
    {
    case frmOpenEvent:
      FrmDrawForm (form);
      FormSetIntField (form, MaxRoachesField, maxRoaches);
      FormSetIntField (form, MinRoachesField, minRoaches);
      FormSetIntField (form, SpeedField, (int)roachSpeed);
      FormSetIntField (form, TurnSpeedField, turnSpeed);
      FrmSetControlValue (form, FrmGetObjectIndex (form, SquishCheckBox),
			  squishRoach);
      processed = true;
      break;
    case ctlSelectEvent:
      switch (event->data.ctlSelect.controlID)
	{
	case OKButton:
	  FormGetIntField (form, MaxRoachesField, &value);
	  if (value < 1)
	    error = true;
	  else
	    maxRoaches = value;
	  FormGetIntField (form, MinRoachesField, &value);
	  if (value > maxRoaches || value < 0)
	    error = true;
	  else
	    minRoaches = value;
	  FormGetIntField (form, SpeedField, &value);
	  if (value <= 0)
	    error = true;
	  else
	    roachSpeed = (float)value;
	  FormGetIntField (form, TurnSpeedField, &value);
	  if (value <= 0)
	    error = true;
	  else
	    turnSpeed = value;
	  squishRoach = FrmGetControlValue
	    (form, FrmGetObjectIndex (form, SquishCheckBox));
	  if (error)
	    FrmAlert (InvalidArgumentAlert);
	case CancelButton:
	  FormFreeField (form, MaxRoachesField);
	  FormFreeField (form, MinRoachesField);
	  FormFreeField (form, SpeedField);
	  FormClose ();
	  processed = true;
	  break;
	}
      break;
    default:
      break;
    }
  return processed;
}

static void
InitRoachForm (form)
     FormPtr form;
{
  FormGadgetType *gadget;
  FrmSetActiveForm (form);
  gadget = FrmGetObjectPtr (form, FrmGetObjectIndex (form, RoachGadget));
  screenX = gadget->rect.topLeft.x;
  screenY = gadget->rect.topLeft.y;
  screenWidth = gadget->rect.extent.x;
  screenHeight = gadget->rect.extent.y;
  while (curRoaches < maxRoaches)
    AddRoach ();
  FrmSetEventHandler (form, (FormEventHandlerPtr)RoachHandleEvent);
  /* Set extended flag, otherwise our gadget handler will not be stored. */
  gadget->attr.extended = true;
  gadget->attr.visible = true;
}

static void
RoachHandleTimeout ()
{
  MenuBarPtr menu = MenuGetActiveMenu ();
  if (menu && menu->attr.visible)
    return;
  if (curRoaches < minRoaches)
    {
      int supplement = RandInt (maxRoaches - curRoaches);
      while (supplement--)
	AddRoach ();
    }
  MoveRoaches ();
  DrawRoaches ();
}

static void
ApplicationDisplayDialog (formID)
     UInt16 formID;
{
  FormActiveStateType current;
  EventType event;
  UInt16 error;
  Boolean keepFormOpen;

  FrmSaveActiveState (&current);
  MemSet (&event, sizeof (EventType), 0);

  event.eType = frmLoadEvent;
  event.data.frmLoad.formID = formID;
  EvtAddEventToQueue (&event);

  event.eType = frmOpenEvent;
  event.data.frmLoad.formID = formID;
  EvtAddEventToQueue (&event);

  keepFormOpen = true;
  while (keepFormOpen)
    {
      EvtGetEvent (&event, evtWaitForever);
      keepFormOpen = (event.eType != frmCloseEvent);
      if (SysHandleEvent (&event))
	continue;
      if (MenuHandleEvent (0, &event, &error))
	continue;
      if (ApplicationHandleEvent (&event))
	continue;
      FrmDispatchEvent (&event);

      if (event.eType == appStopEvent)
	{
	  keepFormOpen = false;
	  EvtAddEventToQueue (&event);
	}
    }
  FrmRestoreActiveState (&current);
}

static Boolean
ApplicationHandleEvent (event)
     EventPtr event;
{
  Boolean processed = false;
  if (event->eType == frmLoadEvent)
    {
      int formID = event->data.frmLoad.formID;
      FormPtr form = FrmInitForm (formID);
      FrmSetActiveForm (form);
      switch (formID)
	{
	case RoachForm:
	  InitRoachForm (form);
	  processed = true;
	  break;
	case PrefsForm:
	  FrmSetEventHandler (form, (FormEventHandlerPtr)PrefsHandleEvent);
	  processed = true;
	  break;
	default:
	  break;
	}
    }
  return processed;
}

static void  
EventLoop ()
{
  EventType event;
  UInt16 error;
  UInt32 end = 0;
  Int32 timeout = evtWaitForever;
  do {
    if (timeoutPeriod == 0)
      timeout = evtWaitForever;
    else
      {
	/* Recalculate absolute end of timeout, since the timeout has
	   just been enabled. */
	if (timeout == evtWaitForever)
	  {
	    end = TimGetTicks () + timeoutPeriod;
	    timeout = timeoutPeriod;
	  }
	else
	  {
	    timeout = end - TimGetTicks ();
	    /* If timeout end already has passed, calculate next
	       timeout and call timeout function. */
	    if (timeout < 0)
	      {
		end = TimGetTicks () + timeoutPeriod;
		if (timeoutHandler)
		  timeoutHandler ();
		continue;
	      }
	  }
      }
    EvtGetEvent (&event, timeout);
    /* Timeout, or possibly some other nil event. */
    if (event.eType == nilEvent)
      {
	/* If the timeout really have expired, calculate next timeout
	   and call timeout function. */
	if (timeoutPeriod != 0 && end <= TimGetTicks ())
	  {
	    end = TimGetTicks () + timeoutPeriod;
	    if (timeoutHandler)
	      timeoutHandler ();
	  }
	/* Always loop around, as we are not interested in feeding
	   this event to the event loop. */
	continue;
      }
    if (SysHandleEvent (&event))
      continue;
    if (MenuHandleEvent (0, &event, &error))
      continue;
    if (ApplicationHandleEvent (&event))
      continue;
    FrmDispatchEvent (&event);
  } while (event.eType != appStopEvent);
}

UInt32
PilotMain (cmd, cmdPBP, launchFlags)
     UInt16 cmd;
     void *cmdPBP;
     UInt16 launchFlags;
{
  if (cmd == sysAppLaunchCmdNormalLaunch)
    {
      UInt32 romVersion35 =
	sysMakeROMVersion (3, 5, 0, sysROMStageRelease, 0);
      UInt32 romVersion;
      FtrGet (sysFtrCreator, sysFtrNumROMVersion, &romVersion);
      supportsRomVersion35 = (romVersion >= romVersion35);
      LoadMathLib ();
      roaches = (Roach *)MemPtrNew (sizeof (Roach) * maxRoaches);
      LoadRoaches ();
      FrmGotoForm (RoachForm);
      EventLoop ();
    }
  return 0;
}
