/*
 *  Stwmcmd - xcomm.c
 *
 *	communicate with Stwm via the root window property.
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <X11/Xatom.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "stwmcmd.h"
#include "parseopt.h"


/**  Global variables
 **/
char*			dpyname = NULL;
int			stwm_windownum;
windowid		stwm_windowid[STWM_MAX_WINDOWS];

static	Display*       	dpy;	/* X display */
static	Window		root;	/* Root window */
static	Atom		XA_STWM_WINDOW_LIST; /* Atom to get a window list */
static	Atom		XA_STWM_COMMAND; /* Atom to put a command */
static	Atom		XA_STWM_RESULT;	/* Atom to get a result */

static	optrec stwm_rsrcs[] = {
    { NULL,		OPT_NULL,	NULL }
};

/* buffers */
static	char		stwm_sendbuf[STWM_BUFSIZ];
static	char*		stwm_sendptr;
static	char		stwm_recvbuf[STWM_BUFSIZ];
static	char*		stwm_recvptr;


/**  Primitive communication with X
 **/

/*  SetStringProp
 *	Primitive routine to set a property of the root window.
 */
static void SetStringProp(Atom atom)
{
    XTextProperty xtp;
    
    if (stwm_sendbuf != stwm_sendptr)
	stwm_sendptr--;
    *stwm_sendptr = '\0';

#ifdef XCOMM_DEBUG
    error("SetStringProp: send: %s", stwm_sendbuf);
#endif

    xtp.encoding = XA_STRING;
    xtp.format = 8;
    xtp.nitems = strlen(stwm_sendbuf);
    xtp.value = (unsigned char*)stwm_sendbuf;
    XSetTextProperty(dpy, root, &xtp, atom);
    stwm_sendptr = stwm_sendbuf;
    *stwm_sendptr = '\0';
}


/*  GetStringProp
 *	Primitive routine to get a property from the root window.
 */
static int GetStringProp(Atom atom)
{
    XTextProperty xtp;
    
    if (XGetTextProperty(dpy, root, &xtp, atom) == 0 ||
	xtp.encoding != XA_STRING ||
	xtp.format != 8)
	return(0);		/* ZERO: error */
    if (STWM_BUFSIZ-1 < xtp.nitems)
	xtp.nitems = STWM_BUFSIZ-1;
    strncpy(stwm_recvbuf, (char*)xtp.value, xtp.nitems);
    stwm_recvbuf[xtp.nitems] = '\0';
    stwm_recvptr = stwm_recvbuf;
    XFree(xtp.value);

#ifdef XCOMM_DEBUG
    error("GetStringProp: recv: %s", stwm_recvbuf);
#endif

    return(1);			/* NON-ZERO: ok */
}


/**  Buffer operation routines
 **/

/*  CheckEof
 *	Return true if the receive buffer pointer is on the EOF.
 */	
int CheckEof(void)
{
    return(! *stwm_recvptr);
}


/*  Skip
 *	Make the buffer pointer skip to the next item.
 */
void Skip(void)
{
    char c;
    while((c = *stwm_recvptr) && c != STWM_DELIMITOR)
	stwm_recvptr++;
    if (c == STWM_DELIMITOR)
	stwm_recvptr++;
}


/*  GetSigned
 *	Get a signed number from the buffer and go next.
 */
long GetSigned(void)
{
    long val = strtol(stwm_recvptr, &stwm_recvptr, 10);
    Skip();
    return(val);
}


/*  GetUnsigned
 *	Get an unsigned number from the buffer and go next.
 */
unsigned long GetUnsigned(void)
{
    unsigned long val = strtoul(stwm_recvptr, &stwm_recvptr, 10);
    Skip();
    return(val);
}


/*  GetString
 *	Get a string from the buffer and go next.
 */
void GetString(char* dest)
{
    char c;
    int i = 1;
    while((c = (*dest = *stwm_recvptr)) &&
	  c != STWM_DELIMITOR &&
	  i < STWM_MAX_STRING) {
	dest++;
	stwm_recvptr++;
    }
    *dest = '\0';
    Skip();
}


/*  GetWinInfo
 *	Get an window information from the buffer.
 *	Wininfo consists of following items in this order:
 *		id		(unsigned)
 *		full name	(string)
 *		title		(string)
 *		icon name	(string)
 *		resource name	(string)
 *		resource class	(string)
 *		mapped?		(signed)
 *		iconified?	(signed)
 *		auto-raised?	(signed)
 *		highlighted?	(signed)
 *		zoomed?		(signed)
 *		width		(unsigned) in pixel
 *		height		(unsigned) in pixel
 *		minimal width	(unsigned) in pixel
 *		minimal height	(unsigned) in pixel
 *		maximum width	(unsigned) in pixel
 *		maximum height	(unsigned) in pixel
 *		base width	(unsigned) in pixel
 *		base height	(unsigned) in pixel
 *		width step	(unsigned) in pixel
 *		height step	(unsigned) in pixel
 *		x of top-left	(unsigned)
 *		y of top-left	(unsigned)
 */
void GetWinInfo(wspec* ws)
{
    char* p = stwm_recvptr;
    ws->id = GetUnsigned();
    GetString(ws->full_name);
    GetString(ws->title);
    GetString(ws->icon_name);
    GetString(ws->res_name);
    GetString(ws->res_class);
    ws->mapped = GetSigned();
    ws->icon   = GetSigned();
    ws->auto_raise = GetSigned();
    ws->highlight = GetSigned();
    ws->zoomed = GetSigned();
    ws->width  = GetUnsigned();
    ws->height = GetUnsigned();
    ws->minwid = GetUnsigned();
    ws->minhgt = GetUnsigned();
    ws->maxwid = GetUnsigned();
    ws->maxhgt = GetUnsigned();
    ws->bwid   = GetUnsigned();
    ws->bhgt   = GetUnsigned();
    ws->widinc = GetUnsigned();
    ws->hgtinc = GetUnsigned();
    ws->x = GetUnsigned();
    ws->y = GetUnsigned();
}


/*  PutSigned
 *	Add a signed number to the end of send buffer.
 */
void PutSigned(long val)
{
    stwm_sendptr += sprintf(stwm_sendptr, "%ld", val);
    *stwm_sendptr++ = STWM_DELIMITOR;
}


/*  PutUnsigned
 *	Add an unsigned number to the end of send buffer.
 */
void PutUnsigned(unsigned long val)
{
    stwm_sendptr += sprintf(stwm_sendptr, "%lu", val);
    *stwm_sendptr++ = STWM_DELIMITOR;
}


/*  PutString
 *	Add a string to the end of send buffer.
 */
void PutString(char* src)
{
    strcpy(stwm_sendptr, src);
    stwm_sendptr += strlen(src);
    *stwm_sendptr++ = STWM_DELIMITOR;
}


/*  ResetSendBuff
 *	Make the send buffer empty.
 */
void ResetSendBuff(void)
{
    stwm_sendptr = stwm_sendbuf;
}


/*  ResetRecvBuff
 *	Make the receive buffer empty.
 */
void ResetRecvBuff(void)
{
    stwm_recvptr = stwm_recvbuf;
}


/**  Initialize, Communicate
 **/

/*  InitX
 *	Initialize X, Atoms and buffer.
 */
void InitX(void)
{
    if (!(dpy = XOpenDisplay(dpyname))) {
	error("can't open display `%s'.", dpyname);
	exit(1);
    }
    root = DefaultRootWindow(dpy);
    getXrsrc(dpy, stwm_rsrcs);
    
    XA_STWM_WINDOW_LIST = XInternAtom(dpy, "STWM_WINDOW_LIST", False);
    XA_STWM_COMMAND = XInternAtom(dpy, "STWM_COMMAND", False);
    XA_STWM_RESULT = XInternAtom(dpy, "STWM_RESULT", False);

    stwm_sendptr = stwm_sendbuf;
}


/*
 *    Stwmcmd uses three properties of the root window to communicate
 *  with Stwm.  Each property is named STWM_WINDOW_LIST, STWM_COMMAND,
 *  STWM_RESULT and indicated with atoms. STWM_WINDOW_LIST is updated
 *  by Stwm when an window created/destroyed. STWM_COMMAND contains
 *  a command by Stwmcmd and when it changes, Stwm notices, receive
 *  that, and put the result of the command into STWM_RESULT.
 */

/*  GetWindowIDList
 *	Get the list of window IDs.
 */
int GetWindowIDList(void)
{
    int i = 0;
    
    if (!GetStringProp(XA_STWM_WINDOW_LIST)) {
	error("can't get window list, is stwm running?");
	exit(1);
    }
    while(!CheckEof() && i < STWM_MAX_WINDOWS)
	stwm_windowid[i++] = GetUnsigned();
    return(stwm_windownum = i);
}


/*  CommunicateStwm
 *	Flush the send buffer and receive the result.
 */
int CommunicateStwm(void)
{
    XEvent xev;
    
    SetStringProp(XA_STWM_COMMAND);
    XSelectInput(dpy, root, PropertyChangeMask);
    do {
	/* wait until receiving the result */
	XNextEvent(dpy, &xev);
    } while(xev.xproperty.atom != XA_STWM_RESULT ||
	    xev.xproperty.state != PropertyNewValue ||
	    !GetStringProp(XA_STWM_RESULT));
    
    return(GetSigned());
}
