/*
 *  parseopt.c
 */

#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

#include "parseopt.h"

static char* progname = NULL;
static FILE* logfile = stderr;

#define ISSPACE(x) ((x) == ' ' || (x) == '\t')
#define ISQUOTE(x) ((x) == quote)
#define ISSEP(x)   ((x) == separator)

/* setwordstr */
static int strgetc(WREC* wr)
{
    char* s = wr->fp;
    char c;
    if (!(c = *s++))
	return(EOF);
    wr->fp = s;
    return(c);
}
static void strungetc(int c, WREC* wr)
{
    char* s = wr->fp;
    *(--s) = (char)c;
    wr->fp =s;
}
void setwordstr(WREC* wr, char* s)
{
    setwordio(wr, strgetc, strungetc, s);
}

void setwordopt(WREC* wr, int q, int s, int c)
{
    wr->quote = q;
    wr->separator = s;
    wr->crstop = c;
}

void setwordio(WREC* wr, getfunc mygetc, ungetfunc myungetc, void* fp)
{
    wr->fp = fp;
    wr->wrgetc = mygetc;
    wr->wrungetc = myungetc;
}

int getword(WREC* wr, char* buff, int count)
{
    int c;
    int initial = 1;
    int fin	= 0;
    int len     = 0;
    int inquote = 0;
    int esc     = 0;
    
    int quote     = wr->quote;
    int separator = wr->separator;
    int crstop    = wr->crstop;
    getfunc mygetc   = wr->wrgetc;
    ungetfunc myungetc = wr->wrungetc;

    count--;
    while( 0 <= (c = mygetc(wr)) ) {
	if (fin) {
	    if (ISSPACE(c))
		c = 0;
	    else if (ISSEP(c))
		break;
	    else if (c != '\n' || crstop) {
		myungetc(c, wr);
		break;
	    } else
		break;
	} else if (esc) {
	    esc = 0;
	    switch(c) {
	    case '\n':
		c = 0;
		break;
	    case 'n':
		c = '\n';
		break;
	    case 't':
		c = '\t';
		break;
	    }
	} else if (c == '\\') {
	    esc = 1;
	    c = 0;
	} else if (c == '\n') {
	    if (crstop) {
		myungetc(c, wr);
		if (initial)
		    len = WORD_CR;
	    }
	    break;
	} else if (initial) {
	    if (c == separator)
		break;
	    else if (ISSPACE(c))
		c = 0;
	    else {
		if (ISQUOTE(c)) {
		    inquote = 1;
		    c = 0;
		}
		initial = 0;
	    }
	} else if (inquote) {
	    if (ISQUOTE(c)) {
		fin = 1;
		c = 0;
	    }
	} else {
	    if (ISQUOTE(c)) {
		myungetc(c, wr);
		break;
	    } else if (ISSPACE(c)) {
		fin = 1;
		c = 0;
	    } else if (c == separator)
		break;
	}
	if (c && count) {
	    *buff++ = c;
	    count--;
	    len++;
	}
    }
    *buff = '\0';
    if (c < 0 && len == 0)
	len = WORD_EOF;
    return(len);
}

int nextline(WREC* wr)
{
    int c;
    getfunc mygetc = wr->wrgetc;
    while((c = mygetc(wr)) != '\n')
	if (c < 0)
	    return(-1);
    return(0);
}

int countitem(char** items)
{
    char** p;
    int i;
    if (!items)
	return(0);
    for(i = 0, p = items; i < MAXITEM; i++, p++)
	if (!*p)
	    break;
    return(i);
}

static char** additem(char** items, char* add)
{
    char** p;
    int i;
    if (!items) {
	p = items = malloc(sizeof(char**) * MAXITEM);
	for(i = 0; i < MAXITEM; i++)
	    *p++ = NULL;
    }
    for(i = 0, p = items; i < MAXITEM-1; i++, p++)
	if (!*p) {
	    *p = strdup(add);
	    break;
	}
    return(items);
}

char** addnitems(char** items, char* s)
{
    char buf[256];
    WREC wr;
    setwordstr(&wr, s);
    setwordopt(&wr, -1, ',', 0);
    while(0 < getword(&wr, buf, sizeof(buf)))
	items = additem(items, buf);
    return(items);
}

char** removeitem(char** items, char* rmv)
{
    char** p;
    int i;
    for(i = 0, p = items; i < MAXITEM; i++, p++)
	if (*p == rmv) {
	    for(; i < MAXITEM; i++, p++)
		p[0] = p[1];
	    p[MAXITEM-1] = NULL;
	    break;
	}
    return(items);
}

char* getprogname(void)
{
    return(progname);
}

void openlogfile(char* file)
{
    if (!(logfile = fopen(file, "a")))
	logfile = stderr;
}

/*VARARGS1*/
void error(char* format, ...)
{
    va_list args;
    if (progname)
	fprintf(logfile, "%s: ", progname);
    va_start(args, format);
    vfprintf(logfile, format, args);
    va_end(args);
    fprintf(logfile, "\n");
}

int strdiff(char* str1, char* str2)
{
    int ind = 0;
    while(*str1 && *str2 && *str1 == *str2)
	str1++, str2++, ind++;
    return(ind);
}

static void parseval(optrec* opt, char* arg)
{
    switch(opt->type) {
    case OPT_BOOLEAN:		/* BOOLEAN */
	if (arg == NULL || arg[0] == 't' || arg[0] == 'T') {
	    *(int*)(opt->ptr) = 1;
	} else if (arg[0] == 'f' || arg[0] == 'F') {
	    *(int*)(opt->ptr) = 0;
	}
	break;
    case OPT_INT:		/* INT */
	*(int*)(opt->ptr) = atoi(arg);
	break;
    case OPT_STRING:		/* STRING */
	*(char**)(opt->ptr) = arg;
	break;
    case OPT_LIST:
	*(char***)(opt->ptr) = addnitems(*(char***)(opt->ptr), arg);
	break;
    }
}


int parseopt(optrec* opts, int argc, char* argv[], int replace)
{
    int numopts = 0;
    int newargc = 0;
    char **newargv0, **newargv1;

    newargv0 = newargv1 = (char**)malloc(sizeof(char*) * (argc+1));
    {
	char c, *s = progname = argv[0];
	while(c = *(s++))
	    if (c == '/') progname = s;
    }
    {
	optrec* opt1 = opts;
	for(numopts = 0; opt1->name && opt1->type; numopts++)
	    opt1++;
    }
    {
	int optmode = 1;
	optrec* needargopt = NULL;
	int i;
	for(i = 1; i < argc; i++) {
	    char* arg1 = argv[i];
	    if (arg1[0] == '-' && optmode) {
		if (needargopt) {
		    error("need value: option -%s", needargopt->name);
		    usage();
		    exit(1);
		}
		if ( ! *++arg1 ) {
		    optmode = 0;
		    continue;
		}
		{
		    int j;
		    optrec* opt1 = opts;
		    for(j = 0; j < numopts; j++, opt1++) {
			int ind = strdiff(opt1->name, arg1);
			if (!opt1->name[ind]) {
			    if (!arg1[ind]) {
				if (opt1->type == OPT_BOOLEAN)
				    parseval(opt1, NULL);
				else
				    needargopt = opt1;
				break;
			    } else {
				if (arg1[ind] == '=') {
				    parseval(opt1, arg1+ind+1);
				    break;
				}
			    }
			}
		    }
		    if (numopts <= j) {
			error("unknown option: -%s", arg1);
			usage();
			exit(1);
		    }
		}
	    } else {
		if (needargopt) {
		    parseval(needargopt, arg1);
		    needargopt = NULL;
		} else {
		    optmode = 0;
		    newargc++;
		    *(newargv1++) = strdup(arg1);
		}
	    }
	}
	if (needargopt) {
	    error("need value: option -%s", needargopt->name);
	    usage();
	    exit(1);
	}
    }
    *newargv1 = NULL;
    if (replace) {
	newargv1 = newargv0;
	while(*argv++ = *newargv1++)
	    ;
	argc = newargc;
    }
    free(newargv0);

    return(argc);
}

void getXrsrc(Display* disp, optrec* rsrc)
{
    while(rsrc->name && rsrc->type) {
	parseval(rsrc, XGetDefault(disp, progname, rsrc->name));
	rsrc++;
    }
}
