/*
 *  Stwmcmd - cmd.c
 *
 *	Main routine
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <X11/Intrinsic.h>	/* for XtOffsetOf macro */

#include "stwmcmd.h"


/**  Macros
 **/

/*  Convert a pixel-based size into letter-based one
 */
#define TWIDTH(w,x)	(((x) - (w)->bwid)/(w)->widinc)
#define THEIGHT(w,y)	(((y) - (w)->bhgt)/(w)->hgtinc)


/**  Static variables
 **/

/*  For query command.
 *  Associate field names to actual member.
 */
#define NUM_FIELDS (sizeof(query_fields)/sizeof(struct qspec))

struct qspec {
    char* name;
    int frmt;			/* 0:int, 1:long, 2:string */
    int ofst;
};

static struct qspec query_fields[] = {
    { "id",		1, XtOffsetOf(wspec, id) },
    { "full_name",	2, XtOffsetOf(wspec, full_name) },
    { "title",		2, XtOffsetOf(wspec, title) },
    { "icon_name",	2, XtOffsetOf(wspec, icon_name) },
    { "resource_name",	2, XtOffsetOf(wspec, res_name) },
    { "resource_class",	2, XtOffsetOf(wspec, res_class) },
    { "mapped",		0, XtOffsetOf(wspec, mapped) },
    { "iconified",	0, XtOffsetOf(wspec, icon) },
    { "auto_raise",	0, XtOffsetOf(wspec, auto_raise) },
    { "highlight",	0, XtOffsetOf(wspec, highlight) },
    { "zoomed",		0, XtOffsetOf(wspec, zoomed) },
    { "left",		0, XtOffsetOf(wspec, x) },
    { "top",		0, XtOffsetOf(wspec, y) },
    { "width",		0, XtOffsetOf(wspec, width) },
    { "height",		0, XtOffsetOf(wspec, height) },
    { "base_width",	0, XtOffsetOf(wspec, bwid) },
    { "base_height",	0, XtOffsetOf(wspec, bhgt) },
    { "inc_width",	0, XtOffsetOf(wspec, widinc) },
    { "inc_height",	0, XtOffsetOf(wspec, hgtinc) },
    { "min_width",	0, XtOffsetOf(wspec, minwid) },
    { "min_height",	0, XtOffsetOf(wspec, minhgt) },
    { "max_width",	0, XtOffsetOf(wspec, maxwid) },
    { "max_height",	0, XtOffsetOf(wspec, maxhgt) }
};



/**  Internal routines
 **/

/*  apply_func_window: Apply the function applyfunc to windows
 *                     which is matched to the window specifiers,
 *                     and return the result code of execution.
 */
static int apply_func_window(funcptr applyfunc, char* argv[])
{
    int i, match = 0;
    wspec ws;

    int nw = GetWindowIDList();
    /* for each window */
    for(i = 0; i < nw; i++) {
	int result;
	/* get window information */
	ResetSendBuff();
	PutString("Q");
	PutUnsigned(stwm_windowid[i]);
	if ((result = CommunicateStwm()) != STWM_RESULTCODE_OK)
	    return(result);
	GetWinInfo(&ws);
	/* check if it is matched */
	if (!MatchWindow(&ws)) {
#ifdef DEBUG
	    error("apply_conf_window: match: %lu", stwm_windowid[i]);
#endif
	    match = 1;
	    /* apply the function */
	    if (result = (*applyfunc)(&ws, argv))
		return(result);
	}
    }
    
    if (!match)			/* nothing matched */
	return(STWM_RESULTCODE_NOWINDOW);
    
    return(STWM_RESULTCODE_OK);
}


/*  print_field: Co-routine of stwm_query.
 *               Display the window spec as appropriate type.
 */
static void print_field(wspec* ws, int i)
{
    char* p = (char*)ws + query_fields[i].ofst;

    /* Type is described in frmt member of struct qspec. */
    switch(query_fields[i].frmt) {
    case 0:			/* unsigned int */
	printf("%u\n", *(unsigned int*)p);
	break;
    case 1:			/* unsigned long */
	printf("%lu\n", *(unsigned long*)p);
	break;
    case 2:			/* string */
	printf("%s\n", p);
	break;
    }
}


/*  stwm_query: Query command.
 *              Display the information of the specified windows.
 */
static int stwm_query(wspec* ws, char* argv[])
{
    int i;

    if (!argv[0]) {
	/* with no argument */
	for(i = 0; i < NUM_FIELDS; i++) {
	    printf("%16s : ", query_fields[i].name);
	    print_field(ws, i);
	}
    } else {
	while(*argv) {
	    for(i = 0; i < NUM_FIELDS; i++) {
		if (!strcmp(*argv, query_fields[i].name)) {
		    print_field(ws, i);
		    break;
		}
	    }
	    argv++;
	}
    }

    return(0);
}


/*  stwm_warp: Warp command.
 *             Make the mouse pointer jump to the specified window.
 */
static int stwm_warp(wspec* ws, char* argv[])
{
    ResetSendBuff();
    PutString("W");
    PutUnsigned(ws->id);
    
    return(CommunicateStwm());
}


/*  stwm_stack: Stack command.
 *              Change order of the specified windows.
 */
static int stwm_stack(wspec* ws, char* argv[])
{
    ResetSendBuff();
    PutString("S");
    PutUnsigned(ws->id);

    PutSigned((argv[0])? atoi(argv[0]) : 1);
    
    return(CommunicateStwm());
}


/*  stwm_zoom: Zoom command.
 *             Change the state of the specified windows.
 */
static int stwm_zoom(wspec* ws, char* argv[])
{
    ResetSendBuff();
    PutString("Z");
    PutUnsigned(ws->id);
    
    PutSigned((argv[0])? atoi(argv[0]) : !ws->icon);
    
    return(CommunicateStwm());
}


/*  stwm_iconfiy: Iconfiy command.
 *                Change the state of the specified windows.
 */
static int stwm_iconify(wspec* ws, char* argv[])
{
    ResetSendBuff();
    PutString("I");
    PutUnsigned(ws->id);

    PutSigned((argv[0])? atoi(argv[0]) : !ws->icon);
    
    return(CommunicateStwm());
}


/*  conf0: Co-routine
 */
static int conf0(char** argp, int oldval, int* newval, int* neg, int rel)
{
    char* p = *argp;
    
#ifdef DEBUG
    error("conf0: arg=%s, oldval=%d, rel=%d", p, oldval, rel);
#endif
    *neg = 0; 
    *newval = oldval;

    if (rel) {
	if (p[0] == '+' || p[0] == '-' || isdigit(p[0])) {
	    *newval += (int)strtol(p, &p, 10);
	} else {
	    return(-1);
	}
    } else {
	if (p[0] == '-') {
	    *neg = 1;
	    *newval = (int)strtol(p, &p, 10);
	} else if (p[0] == '+' || isdigit(p[0])) {
	    *neg = 0;
	    *newval = (int)strtol(p, &p, 10);
	} else {
	    return(-1);
	}
    }
    
    *argp = p;
    return(0);
}


/*  getloc:
 */
static int getloc(char** argp, int* xval, int* xneg, int* yval, int* yneg)
{
    char* p = *argp;
    
#ifdef DEBUG
    error("getloc: arg=%s", p);
#endif

    if (*p != '+' && *p != '-')
	return(-1);
    *xneg = (*p == '-');
    if (!isdigit(p[1]))
	return(-1);
    *xval = (int)strtol(p, &p, 10);
    if (*p != '+' && *p != '-')
	return(-1);
    *yneg = (*p == '-');
    if (!isdigit(p[1]))
	return(-1);
    *yval = (int)strtol(p, &p, 10);

    *argp = p;
    return(0);
}


/*  getsize:
 */
static int getsize(char** argp, int* wval, int* hval)
{
    char* p = *argp;
    
#ifdef DEBUG
    error("getsize: arg=%s", p);
#endif

    *wval = (int)strtol(p, &p, 10);
    if (*p != 'X' && *p != 'x')
	return(-1);
    p++;
    if (!isdigit(*p))
	return(-1);
    *hval = (int)strtol(p, &p, 10);

    *argp = p;
    return(0);
}


/*  stwm_configure:
 */
static int stwm_configure(wspec* ws, char* argv[])
{
    char c;
    char* arg = argv[0];
    int loop = 1;
    int xchange = 0, xneg = 0, xvalue = 0;
    int ychange = 0, yneg = 0, yvalue = 0;
    int wchange = 0, wneg = 0, wvalue = 0;
    int hchange = 0, hneg = 0, hvalue = 0;

    if (!arg)
	return(STWM_RESULTCODE_SYNTAX);

    while(loop) {
#ifdef DEBUG
	error("configure: arg=\"%s\"", arg);
#endif
	c = *arg++;
	switch(c) {
	case '+': case '-':
	    arg--;
	    if (getloc(&arg, &xvalue, &xneg, &yvalue, &yneg))
		loop = 0;
	    else
		xchange = ychange = 1;
	    break;
	case 'w': case 'W':
	    if (conf0(&arg, TWIDTH(ws, ws->width),
		      &wvalue, &wneg, c == 'W'))
		loop = 0;
	    else
		wchange = 1;
	    break;
	case 'h': case 'H':
	    if (conf0(&arg, THEIGHT(ws, ws->height),
		      &hvalue, &hneg, c == 'H'))
		loop = 0;
	    else
		hchange = 1;
	    break;
	case 'x': case 'X':
	    if (conf0(&arg, ws->x, &xvalue, &xneg, c == 'X'))
		loop = 0;
	    else
		xchange = 1;
	    break;
	case 'y': case 'Y':
	    if (conf0(&arg, ws->y, &yvalue, &yneg, c == 'Y'))
		loop = 0;
	    else
		ychange = 1;
	    break;
	default:
	    if (isdigit(c)) {
		arg--;
		if (getsize(&arg, &wvalue, &hvalue)) {
		    loop = 0;
		    break;
		}
		wchange = hchange = 1;
		if (*arg == '+' || *arg == '-') {
		    if (getloc(&arg, &xvalue, &xneg, &yvalue, &yneg))
			loop = 0;
		    else
			xchange = ychange = 1;
		}
	    } else {
		loop = 0;
	    }
	    break;
	}
    }

    if (!(xchange || ychange || wchange || hchange))
	return(STWM_RESULTCODE_SYNTAX);

    if (hchange) {
	if (THEIGHT(ws, ws->maxhgt) < hvalue)
	    hvalue = THEIGHT(ws, ws->maxhgt);
	if (hvalue < THEIGHT(ws, ws->minhgt))
	    hvalue = THEIGHT(ws, ws->minhgt);
    }
    if (wchange) {
	if (TWIDTH(ws, ws->maxwid) < wvalue)
	    wvalue = TWIDTH(ws, ws->maxwid);
	if (wvalue < TWIDTH(ws, ws->minwid))
	    wvalue = TWIDTH(ws, ws->minwid);
    }

#ifdef DEBUG
    if (xchange) error("configure: xchange: %c %d", xneg? '-' : '+', xvalue);
    if (ychange) error("configure: ychange: %c %d", yneg? '-' : '+', yvalue);
    if (wchange) error("configure: wchange: %d", wvalue);
    if (hchange) error("configure: hchange: %d", hvalue);
#endif

    ResetSendBuff();
    PutString("C");
    PutUnsigned(ws->id);
    
    PutSigned(xchange);
    PutSigned(xneg);
    PutSigned(xvalue);

    PutSigned(ychange);
    PutSigned(yneg);
    PutSigned(yvalue);

    PutSigned(wchange);
    PutSigned(wvalue);

    PutSigned(hchange);
    PutSigned(hvalue);

    return(CommunicateStwm());
}


/*  usage: Display help message.
 */
void usage(void)
{
    error("usage: %s [-d display] [-version] wspecs command [args]",
	  getprogname());
    error("  wspec:   -i id1, ..     specify window id");
    error("           -t title1, ..  specify window title");
    error("           -c class1, ..  specify resource class");
    error("           -f             full-string matching");
    error("           -s             case-sensitive matching");
    error("  command: query, configure, stack, zoom, iconify, warp");
    exit(1);
}


/**  Main routine
 **/
int main(int argc, char* argv[])
{
    char** cmdargv;
    int    cmd, result;

    /* Get command and options */
    cmd = GetCommand(argc, argv, &cmdargv);
    
    /* Initialize X */
    InitX();
    
    /* Do command */
    switch(cmd) {
    case CMD_QUERY:
	result = apply_func_window((funcptr)stwm_query, cmdargv);
	break;
    case CMD_WARP:
	result = apply_func_window((funcptr)stwm_warp, NULL);
	break;
    case CMD_STACK:
	result = apply_func_window((funcptr)stwm_stack, cmdargv);
	break;
    case CMD_ZOOM:
	result = apply_func_window((funcptr)stwm_zoom, cmdargv);
	break;
    case CMD_ICONIFY:
	result = apply_func_window((funcptr)stwm_iconify, cmdargv);
	break;
    case CMD_CONFIGURE:
	result = apply_func_window((funcptr)stwm_configure, cmdargv);
	break;
    }
    
    /* Process the result code */
    switch(result) {
    case STWM_RESULTCODE_UNKNOWN:
	error("error: internal error, unknown command #: %d.", cmd);
	exit(1);
	break;
    case STWM_RESULTCODE_NOWINDOW:
	error("error: stwm: no such window.");
	exit(1);
	break;
    case STWM_RESULTCODE_SYNTAX:
	error("error: command syntax error: '%s %s'", argv[0], cmdargv[0]);
	usage();
    }
    
    exit(0);
}
